<?php

namespace App\Services;

use Illuminate\Support\Str;

class FaqService
{
    protected function storagePath(): string
    {
        return storage_path('app/ai/faqs.json');
    }

    /**
     * Return merged FAQs from config and storage (admin-added).
     */
    public function all(): array
    {
        $fromConfig = (array) config('ai_faq');
        $merged = [];
        foreach ($fromConfig as $f) {
            if (is_array($f) && !empty($f['id'])) { $merged[$f['id']] = $f; }
        }
        try {
            $path = $this->storagePath();
            if (file_exists($path)) {
                $raw = (string) file_get_contents($path);
                $arr = json_decode($raw, true);
                if (is_array($arr)) {
                    foreach ($arr as $f) {
                        if (!is_array($f) || empty($f['id'])) { continue; }
                        // Tombstone support: if deleted flag is present, hide from merged
                        if (!empty($f['deleted'])) { unset($merged[$f['id']]); continue; }
                        $merged[$f['id']] = $f;
                    }
                }
            }
        } catch (\Throwable $e) {}
        return array_values($merged);
    }

    /**
     * Find one FAQ by id (merged view). Storage overrides config.
     */
    public function find(string $id): ?array
    {
        $id = trim($id);
        if ($id === '') return null;
        $map = [];
        foreach ((array) config('ai_faq') as $f) {
            if (is_array($f) && ($f['id'] ?? '') !== '') { $map[$f['id']] = $f + ['source' => 'config']; }
        }
        try {
            $path = $this->storagePath();
            if (file_exists($path)) {
                $raw = (string) file_get_contents($path);
                $arr = json_decode($raw, true);
                if (is_array($arr)) {
                    foreach ($arr as $f) {
                        if (!is_array($f) || ($f['id'] ?? '') === '') continue;
                        if (!empty($f['deleted'])) { unset($map[$f['id']]); continue; }
                        $map[$f['id']] = $f + ['source' => 'storage'];
                    }
                }
            }
        } catch (\Throwable $e) {}
        return $map[$id] ?? null;
    }

    /**
     * Create/update an FAQ entry in storage.
     */
    public function upsert(array $data): bool
    {
        $id = (string) ($data['id'] ?? '');
        if ($id === '') return false;
        $existing = [];
        $path = $this->storagePath();
        try {
            if (file_exists($path)) {
                $raw = (string) file_get_contents($path);
                $arr = json_decode($raw, true);
                if (is_array($arr)) { $existing = $arr; }
            }
        } catch (\Throwable $e) {}
        // index by id
        $map = [];
        foreach ($existing as $f) {
            if (is_array($f) && !empty($f['id'])) { $map[$f['id']] = $f; }
        }
        $map[$id] = [
            'id' => $id,
            'patterns' => array_values(array_filter(array_map('trim', (array)($data['patterns'] ?? [])))) ,
            'answer' => (string) ($data['answer'] ?? ''),
            'answer_translations' => (array) ($data['answer_translations'] ?? []),
        ];
        // persist
        $dir = dirname($path);
        if (!is_dir($dir)) @mkdir($dir, 0775, true);
        file_put_contents($path, json_encode(array_values($map), JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
        return true;
    }

    /**
     * Delete an FAQ from storage (does not touch config-defined FAQs).
     */
    public function delete(string $id): bool
    {
        $id = trim($id);
        if ($id === '') return false;
        $path = $this->storagePath();
        $existing = [];
        try {
            if (file_exists($path)) {
                $raw = (string) file_get_contents($path);
                $arr = json_decode($raw, true);
                if (is_array($arr)) { $existing = $arr; }
            }
        } catch (\Throwable $e) {}
        $out = [];
        $foundActiveInStorage = false;
        foreach ($existing as $f) {
            if (!is_array($f) || ($f['id'] ?? '') === '') continue;
            // drop any record with this id (active or tombstone)
            if ((string)$f['id'] === $id && empty($f['deleted'])) { $foundActiveInStorage = true; continue; }
            if ((string)$f['id'] === $id && !empty($f['deleted'])) { continue; }
            $out[] = $f;
        }
        // If there was no active custom entry, create a tombstone to hide config one
        if (!$foundActiveInStorage) {
            $out[] = [ 'id' => $id, 'deleted' => true ];
        }
        $dir = dirname($path);
        if (!is_dir($dir)) @mkdir($dir, 0775, true);
        file_put_contents($path, json_encode(array_values($out), JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
        return true;
    }
    /**
     * Try to find an FAQ answer for a free-form question.
     * Returns string answer or null if no good match.
     */
    public function answerFor(string $question): ?string
    {
        $q = strtolower(trim($question));
        if ($q === '') { return null; }

        // Merge config and stored FAQs
        $faqs = $this->all();
        $bestScore = 0; $best = null;
        // Cue: user asked specifically for estimated delivery time/date
        $wantsEta = (bool) preg_match('/\b(kailan|when|estimated\s+delivery(?:\s+date)?|delivery\s+date|eta|lead\s*time|ilang\s+araw|gaano\s+katagal|kailan\s+ma[iy]deliver)\b/i', $q);
        foreach ($faqs as $faq) {
            $patterns = (array) ($faq['patterns'] ?? []);
            $score = 0;
            foreach ($patterns as $p) {
                $p = strtolower((string) $p);
                if ($p === '') continue;
                // count occurrences for a soft score
                if (Str::contains($q, $p)) { $score += 2; }
                // token overlap boost
                $tokens = preg_split('/[^a-z0-9]+/i', $p);
                $hits = 0; $need = 0;
                foreach ($tokens as $t) {
                    $t = trim($t);
                    if (strlen($t) < 3) continue;
                    $need++;
                    if (Str::contains($q, $t)) { $hits++; }
                }
                if ($need > 0 && $hits >= max(1, (int) ceil($need * 0.6))) {
                    $score += 1;
                }
            }
            // Disambiguation: if asking for ETA, prefer ETA over generic delivery
            if ($wantsEta && (($faq['id'] ?? '') === 'eta')) { $score += 5; }
            if ($wantsEta && (($faq['id'] ?? '') === 'delivery')) { $score -= 1; }
            // small preference for broader FAQs
            if ($score > $bestScore) { $bestScore = $score; $best = $faq; }
        }

        // threshold so unrelated queries don't trigger
        if ($best && $bestScore >= 2) {
            // Choose localized answer if available
            $lang = (string) session('ai_chat.lang_pref', '');
            $answer = (string) ($best['answer'] ?? '');
            $translations = (array) ($best['answer_translations'] ?? []);
            // External translation map (config/ai_faq_translations.php) keyed by id
            $extMap = (array) config('ai_faq_translations');
            if (empty($translations) && !empty($best['id']) && isset($extMap[$best['id']]) && is_array($extMap[$best['id']])) {
                $translations = (array) $extMap[$best['id']];
            }
            if ($lang !== '' && isset($translations[$lang]) && is_string($translations[$lang]) && $translations[$lang] !== '') {
                return (string) $translations[$lang];
            }
            return $answer;
        }
        return null;
    }
}
